from itertools import count
from flask import (Blueprint, redirect, render_template, request, session, url_for, jsonify)
from .io import write_data, write_metadata
import json, random
import time, os
import numpy as np

## Initialize blueprint.
bp = Blueprint('experiment', __name__)

@bp.route('/experiment')
def experiment():
    """Present jsPsych experiment to participant."""

    ## Error-catching: screen for missing session.
    if not 'workerId' in session:

        ## Redirect participant to error (missing workerId).
        return redirect(url_for('error.error', errornum=1000))

    ## Case 1: previously completed experiment.
    elif 'complete' in session:

        ## Redirect participant to complete page.
        return redirect(url_for('complete.complete'))

    ## Case 2: repeat visit.
    elif not session['allow_restart'] and 'experiment' in session:

        ## Update participant metadata.
        session['ERROR'] = "1004: Revisited experiment."
        session['complete'] = 'error'
        write_metadata(session, ['ERROR','complete'], 'a')

        ## Redirect participant to error (previous participation).
        return redirect(url_for('error.error', errornum=1004))

    ## Case 3: first visit.
    else:
        
        ## Update participant metadata.
        session['experiment'] = True
        incomp_dir = session['incomplete'].replace('\\\\','\\')
        incomp_file = os.path.join(incomp_dir, session['subId']+".json");
        
        sets = os.listdir(session['imgdir']);
        with open(session['ljson'], "r") as jfile:
            labeled_sets = json.load(jfile)

        choices = set(list(labeled_sets.keys()))
        choice_left = []
        for choice in choices :
            if len(labeled_sets[choice]) < 25 :
                choice_left.append(choice)

        target_set = np.random.choice(choice_left, size=1)[0];
        session['target_set'] = target_set
        labeled_sets[target_set].append(session["workerId"])

        with open(session['ljson'], "w") as jfile :
            json.dump(labeled_sets,jfile)      
            
        session['incomp_file'] = incomp_file
        
        target_imgdir = os.path.join(session['imgdir'], target_set);
        target_imgs = []
        for file in os.listdir(target_imgdir) :
            if file.endswith(".png") :
                target_imgs.append(os.path.join(session['imgdir_rel'],target_set,file))

        session['target_imgs'] = target_imgs
        random.shuffle(target_imgs)
        write_metadata(session, ['experiment'], 'a')

        ## Present experiment.
        return render_template('experiment.html', workerId=session['workerId'], 
                               assignmentId=session['assignmentId'], 
                               hitId=session['hitId'], a=session['a'], 
                               tp_a=session['tp_a'], b=session['b'], 
                               tp_b=session['tp_b'], c=session['c'], 
                               tp_c=session['tp_c'])

@bp.route('/experiment', methods=['POST'])
def pass_message():
    """Write jsPsych message to metadata."""
    
    if request.is_json:

        ## Retrieve jsPsych data.
        msg = request.get_json()

        ## Update participant metadata.
        session['MESSAGE'] = msg
        write_metadata(session, ['MESSAGE'], 'a')

    ## DEV NOTE:
    ## This function returns the HTTP response status code: 200
    ## Code 200 signifies the POST request has succeeded.
    ## For a full list of status codes, see:
    ## https://developer.mozilla.org/en-US/docs/Web/HTTP/Status
    return ('', 200)

@bp.route('/incomplete_save', methods=['POST'])
def incomplete_save():
    """Save incomplete jsPsych dataset to disk."""

    if request.is_json:

        ## Retrieve jsPsych data.
        JSON = request.get_json()

        ## Save jsPsch data to disk.
        write_data(session, JSON, method='incomplete')

    ## Flag partial data saving.
    session['MESSAGE'] = 'incomplete dataset saved'
    write_metadata(session, ['MESSAGE'], 'a')

    ## DEV NOTE:
    ## This function returns the HTTP response status code: 200
    ## Code 200 signifies the POST request has succeeded.
    ## For a full list of status codes, see:
    ## https://developer.mozilla.org/en-US/docs/Web/HTTP/Status
    return ('', 200)

@bp.route('/redirect_success', methods = ['POST'])
def redirect_success():
    """Save complete jsPsych dataset to disk."""

    if request.is_json:

        ## Retrieve jsPsych data.
        JSON = request.get_json()

        ## Save jsPsch data to disk.
        write_data(session, JSON, method='pass')

    ## Flag experiment as complete.
    session['complete'] = 'success'
    write_metadata(session, ['complete'], 'a')

    ## DEV NOTE:
    ## This function returns the HTTP response status code: 200
    ## Code 200 signifies the POST request has succeeded.
    ## The corresponding jsPsych function handles the redirect.
    ## For a full list of status codes, see:
    ## https://developer.mozilla.org/en-US/docs/Web/HTTP/Status
    return ('', 200)

@bp.route('/redirect_reject', methods = ['POST'])
def redirect_reject():
    """Save rejected jsPsych dataset to disk."""

    if request.is_json:

        ## Retrieve jsPsych data.
        JSON = request.get_json()

        ## Save jsPsch data to disk.
        write_data(session, JSON, method='reject')

    ## Flag experiment as complete.
    session['complete'] = 'reject'
    write_metadata(session, ['complete'], 'a')

    ## DEV NOTE:
    ## This function returns the HTTP response status code: 200
    ## Code 200 signifies the POST request has succeeded.
    ## The corresponding jsPsych function handles the redirect.
    ## For a full list of status codes, see:
    ## https://developer.mozilla.org/en-US/docs/Web/HTTP/Status
    return ('', 200)

@bp.route('/redirect_error', methods = ['POST'])
def redirect_error():
    """Save rejected jsPsych dataset to disk."""

    if request.is_json:

        ## Retrieve jsPsych data.
        JSON = request.get_json()

        ## Save jsPsch data to disk.
        write_data(session, JSON, method='reject')

    ## Flag experiment as complete.
    session['complete'] = 'error'
    write_metadata(session, ['complete'], 'a')

    ## DEV NOTE:
    ## This function returns the HTTP response status code: 200
    ## Code 200 signifies the POST request has succeeded.
    ## The corresponding jsPsych function handles the redirect.
    ## For a full list of status codes, see:
    ## https://developer.mozilla.org/en-US/docs/Web/HTTP/Status
    return ('', 200)


@bp.route('/target_imgs', methods=["GET"])
def target_imgs():
    """return target_imgs"""

    return json.dumps({"target_imgs" : session["target_imgs"]})